/* AMX Mod X
*   CTF MOD
*
* (c) Copyright 2006 by VEN aka VnMnL
*
* This file is provided as is (no warranties)
*
*     DESCRIPTION
*       Plugin is the "Opposing Force CTF"-like mod for CS.
*       Get the enemy's flag and return it to your base.
*       If your flag is missed you must take it first.
*
*     MODULES
*       fakemeta
*       engine (only for AMXX < v1.75)
*
*     COMMANDS
*       ctf_init - initializes the CTF mode
*
*     VERSIONS
*       0.3   fixed: defkit model on the CTs back was removed
*             added voting on bomb/non-obj. maps option (ON)
*             implemented "ctf_init" command (ADMIN_MAP)
*             solved class selection block issue (autoclass)
*             changed vote menu item number options (7, 8)
*             added vote delay after game start option (3.0)
*             added an option to log vote results (ON)
*       0.2   cshack module not required anymore (fake score)
*             implemented base stand sticking auto prevention
*             changed game description ("Counter-Strike CTF")
*             backpack model on the T-players back is blocked
*             players money change on draw round is blocked
*             hud text message "you have the bomb" is blocked
*             log message "spawned with the bomb" is blocked
*       0.1   first release
*/

// plugin's main information
#define PLUGIN_NAME "CTF MOD"
#define PLUGIN_VERSION "0.3"
#define PLUGIN_AUTHOR "VEN"

// standard includes
#include <amxmodx> // AMX Mod X required
#include <fakemeta> // fakemeta module required
#include <amxmisc> // this is not a module!

// engine module requirement detection
#if defined AMXX_VERSION_NUM
	#if AMXX_VERSION_NUM < 175 // if AMXX < v1.75
		#include <engine>
	#endif
#else // if AMXX version can't be checked
	#include <engine>
#endif

// OPTIONS BELOW

// comment to disable force team win on flag capture
#define FORCE_TEAM_WIN

// checks whether force team win is enabled and if so includes a native
#if defined FORCE_TEAM_WIN
	#include <ftw> // ftw.inc, includes cs_force_team_win native, uses "Force Team Win"
#endif

// this title will be shown in "Find Servers" and other servers monitoring tools
new g_game_description[] = "Counter-Strike CTF"

// enables voting on bomb/defuse and non-objective maps, comment to disable voting
#define VOTING_ON_BOBM_AND_NONOBJ_MAPS

// name and access level of the command to initialize the CTF mode
// note that the command will work only on bomb/defuse or non-objective maps
#define COMMAND_CTF_INIT "ctf_init"
#define ACCESS_CTF_INIT ADMIN_MAP

// CTF vote menu item numbers
#define KEY_NUM_Y 7
#define KEY_NUM_N 8

// CTF vote Y/N required ratio
#define VOTE_RATIO 0.51

// CTF vote delay after game commencing
#define VOTE_DELAY 3.0

// CTF vote time
#define VOTE_TIME 10.0

// comment to disable logging vote results to the specified file
//#define LOG_FILE "ctf.log"

// set to -1 to disable auto returning the dropped flag to the base
#define FLAG_AUTORETURN_DELAY 30

// frag bonus for the various events
#define BONUS_TAKEN 1
#define BONUS_RETURNED 1
#define BONUS_CAPTURED 2

// OPTIONS ABOVE

// team IDs
#define TEAM_T 1
#define TEAM_CT 2

// used to associate team-specific array elements with a team
enum {
	TA_T,
	TA_CT,
	TA_SIZE
}

// retrieves TA index from TEAM index and vice versa
#define TA_FROM_TEAM(%1) ((%1) - 1)
#define TEAM_FROM_TA(%1) ((%1) + 1)

// Black Messa/Opposing Force flag IDs
#define FLAG_BM 1
#define FLAG_OF 2

// assigns flags to a teams
new g_flag_team[TA_SIZE] = {FLAG_BM, FLAG_OF}

// used for playing flag animation sequences
#define SEQ_BASE 4
#define SEQ_CARRIED 2
#define SEQ_DROPPED 1

// base stand min and max bounds
new Float:g_base_min[3] = {-24.0, -24.0, 0.0}
new Float:g_base_max[3] = {24.0, 24.0, 16.0}

// flag min and max bounds
new Float:g_flag_min[3] = {-16.0, -16.0, 0.0}
new Float:g_flag_max[3] = {16.0, 16.0, 72.0}

// base and flag models
new g_mdl_base[3][] = {"", "models/civ_stand.mdl", "models/mil_stand.mdl"}
new g_mdl_flag[] = "models/flag.mdl"

// take and capture sounds
new g_snd_taken[3][] = {"", "CTF/bm_flagtaken.wav", "CTF/soldier_flagtaken.wav"}
new g_snd_captured[3][] = {"", "CTF/civ_flag_capture.wav", "CTF/marine_flag_capture.wav"}

enum {
	txt_taken,
	txt_dropped,
	txt_returned,
	txt_returned_auto,
	txt_captured,
	txt_num
}

enum {
	dest_player,
	dest_teammates,
	dest_enemies,
	dest_num
}

// text messages
new g_txt[txt_num][dest_num][] = {
	{
		"You've got the enemy flag! Return to base!",
		"Your team GOT the ENEMY flag!!",
		"Your flag has been TAKEN!!"
	},
	{
		"You dropped the enemy flag!",
		"Your team dropped the enemy flag!",
		"Your flag was dropped!"
	},
	{
		"You've returned your flag!",
		"Your flag was returned!",
		"Enemy flag was returned!"
	},
	{
		"",
		"Your flag was returned (auto)!",
		"Enemy flag was returned (auto)!"
	},
	{
		"You CPATURED the flag!",
		"Your team CPATURED the flag!!",
		"Your flag was CPATURED!!"
	}
}

// flag can't be captured because own flag is missed/taken so here is the hint
#define HINT_DELAY 1.0
new g_hint[] = "You must return YOUR flag first!"
new Float:g_next_hint[TA_SIZE]

// reasons of denying execution of the command to initialize the CTF mode
new g_cmd_deny_vote[] = "[CTF] There is currently a voting for the CTF mode are running!"
new g_cmd_deny_init[] = "[CTF] The CTF mode has already been initialized!"

// base and flag classnames
new g_ctfbase[] = "item_ctfbase"
new g_ctfflag[] = "item_ctfflag"

// bomb should't be given so creation of the given classname will be superceded
new g_weapon_c4[] = "weapon_c4"

// used to supercede c4 icon displaying
new g_icon_c4[] = "c4"

// icon names
new g_icon[TA_SIZE][] = {"number_1", "number_2"}

// icon modes
#define ICON_NONE 0
#define ICON_NORMAL (1<<0)
#define ICON_BLINK (1<<1)

enum {
	state_base,
	state_carried,
	state_dropped,
	state_num
}

enum {
	color_red,
	color_green,
	color_blue,
	color_num
}

// icon colors for the various flag states
new g_icon_color[state_num][color_num] = {
	{0, 255, 0},
	{255, 0, 0},
	{255, 255, 0}
}

// part of the corresponding log message what will be blocked
new g_spawned_withbomb[] = "triggered ^"Spawned_With_The_Bomb^""

// this hud message will be blocked
new g_have_bomb[] = "#Hint_you_have_the_bomb"

// game predefined text codes used to block team win messages on flag capture
new g_win_msg[TA_SIZE][] = {"#Terrorists_Win", "#CTs_Win"}

// part of the corresponding log message what will be blocked due to a fake draw
new g_target_saved[] = "Team ^"CT^" triggered ^"Target_Saved^" (CT"

// round draw log message template
new g_round_draw[] = "World triggered ^"Round_Draw^" (CT ^"%d^") (T ^"%d^")"

// game predefined text code used to send round draw message
new g_draw_msg[] = "#Round_Draw"

// game predefined audio code used to send round draw audio
new g_draw_audio[] = "%!MRAD_rounddraw"

// team names
new g_team_name[TA_SIZE][] = {"TERRORIST", "CT"}

// team score will be cached here
new g_team_score[TA_SIZE]

// here will be stored a difirence to do a fake CT score due to a fake draw
new g_ct_scorediff

// player scoreboard attributes
#define ATTRIB_NONE 0
#define ATTRIB_DEAD (1<<0)
#define ATTRIB_BOMB (1<<1)
#define ATTRIB_VIP (1<<2)

// assign scoreboard attributes to a team
new g_attrib[TA_SIZE] = {ATTRIB_BOMB, ATTRIB_VIP}

// CTF vote menu header
new g_menu_header[] = "Enable Capture the Flag (CTF) Mode?"

// CTF vote menu choices
new g_choice_y[] = "Yes"
new g_choice_n[] = "No"

// retrieves key indexes from the specified key numbers
#if KEY_NUM_Y
	#define KEY_IDX_Y (KEY_NUM_Y - 1)
#else
	#define KEY_IDX_Y 9
#endif

#if KEY_NUM_N
	#define KEY_IDX_N (KEY_NUM_N - 1)
#else
	#define KEY_IDX_N 9
#endif

// retrieves keys bit sum
#define KEY_BITSUM ((1<<KEY_IDX_Y) | (1<<KEY_IDX_N))

// whether there game commencing or not will be cached here
new bool:g_game_commencing

// whether there are a voting for the CTF mode or not will be cached here
new bool:g_voting

// whether the CTF mode are initialized or not will be cached here
new bool:g_initialized

// vote results will be stored here
new g_voted_y
new g_voted_n

// msg ids will be cached here
new g_msgid_hudtextargs
new g_msgid_statusicon
new g_msgid_scoreinfo
new g_msgid_scoreattrib
new g_msgid_textmsg
new g_msgid_sendaudio
new g_msgid_teamscore
new g_msgid_money

// max. players will be cached here
new g_maxplayers

// base origin and angles will be cached here
new Float:g_base_origin[TA_SIZE][3]
new Float:g_base_angles[TA_SIZE][3]

// base and flag entids will be cached here
new g_base_ent[TA_SIZE]
new g_flag_ent[TA_SIZE]

// flag state and owner will be cached here
new g_flag_state[TA_SIZE]
new g_flag_owner[TA_SIZE]

// used to determine whether the gained reward needs to be reset
new bool:g_reset_reward

// users money before reward will be cached here
new g_user_money[33]

// uncomment to disable automatic 32/64bit processor detection
// possible values are <0: 32bit | 1: 64bit>
//#define PROCESSOR_TYPE 0

// private data deaths offset
#define OFFSET_DEATHS_32BIT 444
#define OFFSET_DEATHS_64BIT 493

// deaths offset linux difference
#define OFFSET_DEATHS_LINUXDIFF 5

// private data money offset
#define OFFSET_MONEY_32BIT 115
#define OFFSET_MONEY_64BIT 140

// money offset linux difference
#define OFFSET_MONEY_LINUXDIFF 5

// determination of actual offsets
#if !defined PROCESSOR_TYPE // is automatic 32/64bit processor detection?
	#if cellbits == 32 // is the size of a cell are 32 bits?
		// then considering processor as 32bit
		#define OFFSET_DEATHS OFFSET_DEATHS_32BIT
		#define OFFSET_MONEY OFFSET_MONEY_32BIT
	#else // in other case considering the size of a cell as 64 bits
		// and then considering processor as 64bit
		#define OFFSET_DEATHS OFFSET_DEATHS_64BIT
		#define OFFSET_MONEY OFFSET_MONEY_64BIT
	#endif
#else // processor type specified by PROCESSOR_TYPE define
	#if PROCESSOR_TYPE == 0 // 32bit processor defined
		#define OFFSET_DEATHS OFFSET_DEATHS_32BIT
		#define OFFSET_MONEY OFFSET_MONEY_32BIT
	#else // considering that 64bit processor defined
		#define OFFSET_DEATHS OFFSET_DEATHS_64BIT
		#define OFFSET_MONEY OFFSET_MONEY_64BIT
	#endif
#endif

// marco to retrieve user deaths
#define CS_GET_USER_DEATHS_(%1) get_pdata_int(%1, OFFSET_DEATHS, OFFSET_DEATHS_LINUXDIFF)

// macro to get user money
#define CS_GET_USER_MONEY_(%1) get_pdata_int(%1, OFFSET_MONEY, OFFSET_MONEY_LINUXDIFF)

// macro to set user money
#define CS_SET_USER_MONEY_(%1,%2) set_pdata_int(%1, OFFSET_MONEY, %2, OFFSET_MONEY_LINUXDIFF)

// macro to check whether entity with the given classname exists or not
#define IS_ENT_EXISTS(%1) engfunc(EngFunc_FindEntityByString, -1, "classname", %1)

public plugin_precache() {
	for (new i; i < TA_SIZE; ++i) {
		precache_model(g_mdl_base[g_flag_team[i]])
		precache_sound(g_snd_taken[g_flag_team[i]])
		precache_sound(g_snd_captured[g_flag_team[i]])
	}

	precache_model(g_mdl_flag)
}

public plugin_init() {
	register_plugin(PLUGIN_NAME, PLUGIN_VERSION, PLUGIN_AUTHOR)

	if (IS_ENT_EXISTS("hostage_entity") || IS_ENT_EXISTS("func_escapezone") || IS_ENT_EXISTS("func_vip_safetyzone"))
		return

	register_concmd(COMMAND_CTF_INIT, "concmd_ctf_init", ACCESS_CTF_INIT, "- initializes the CTF mode")

#if defined VOTING_ON_BOBM_AND_NONOBJ_MAPS

	register_event("TextMsg", "event_game_commencing", "a", "2=#Game_Commencing")
	register_event("HLTV", "event_new_round_", "a", "1=0", "2=0")

	register_menucmd(register_menuid(g_menu_header), KEY_BITSUM, "vote_recieved")
#endif
}

public event_game_commencing() {
	g_game_commencing = true
}

public event_new_round_() {
	if (!g_initialized && g_game_commencing && !g_voting) {
		g_game_commencing = false
		g_voting = true
		g_voted_y = 0
		g_voted_n = 0
		set_task(VOTE_DELAY, "vote_init")
	}
}

public concmd_ctf_init(id, level, cid) {
	if (!cmd_access(id, level, cid, 1))
		return PLUGIN_HANDLED

	if (g_voting)
		console_print(id, g_cmd_deny_vote)
	else if (g_initialized)
		console_print(id, g_cmd_deny_init)
	else {
		g_initialized = true

		new name[32], authid[32]
		get_user_name(id, name, 31)
		get_user_authid(id, authid, 31)

		log_amx("Cmd: ^"%s<%d><%s><>^" initialized the CTF mode", name, get_user_userid(id), authid)
		
		switch (get_cvar_num("amx_show_activity")) {
			case 2: client_print(0, print_chat, "ADMIN %s: initialized the CTF mode", name)
			case 1: client_print(0, print_chat, "ADMIN: initialized the CTF mode")
		}

		ctf_init()
	}

	return PLUGIN_HANDLED
}

public vote_init() {
	set_cvar_float("amx_last_voting", get_gametime() + VOTE_TIME)

	new msgid = get_user_msgid("TextMsg")
	new block = get_msg_block(msgid)
	set_msg_block(msgid, BLOCK_SET)
	engclient_cmd(0, "joinclass", "5")
	set_msg_block(msgid, block)

	new menu[128]
	format(menu, 127, "^n\y%s^n^n\w%d. %s^n%d. %s^n", g_menu_header, KEY_NUM_Y, g_choice_y, KEY_NUM_N, g_choice_n)
	show_menu(0, KEY_BITSUM, menu, floatround(VOTE_TIME))
	set_task(VOTE_TIME, "vote_result")
}

public vote_recieved(id, key) {
	new name[32]
	get_user_name(id, name, 31)
	new bool:voted_yes = key == KEY_IDX_Y
	client_print(0, print_chat, "%s voted for ^"%s^"", name, voted_yes ? g_choice_y : g_choice_n)
	if (voted_yes)
		g_voted_y++
	else
		g_voted_n++

	return PLUGIN_HANDLED
}

public vote_result() {
	g_voting = false
	new votes = g_voted_y + g_voted_n
	new Float:ratio = votes ? (float(g_voted_y) / float(votes)) : 0.0
	new bool:successful = ratio >= VOTE_RATIO
	client_print(0, print_chat, "Vote for CTF Mode %s. [%s: %d], [%s: %d], [Ratio: %.2f]", successful ? "successful" : "failed", g_choice_y, g_voted_y, g_choice_n, g_voted_n, ratio)
	if (successful) {
		g_initialized = true
		set_task(2.0, "ctf_init")
	}

#if defined LOG_FILE
	new mapname[32]
	get_mapname(mapname, 31)
	log_to_file(LOG_FILE, "%.2f   %2d   %2d   %s", ratio, g_voted_y, g_voted_n, mapname)
#endif
}

public ctf_init() {
	g_maxplayers = get_maxplayers()

	register_event("TextMsg", "event_new_game", "a", "2=#Game_Commencing", "2=#Game_will_restart_in")
	register_event("HLTV", "event_new_round", "a", "1=0", "2=0")
	register_logevent("logevent_round_end", 2, "1=Round_End")
	register_event("ResetHUD", "event_reset_hud", "b")
	register_event("DeathMsg", "event_death", "a", "2!0")

	g_msgid_hudtextargs = get_user_msgid("HudTextArgs")
	g_msgid_statusicon = get_user_msgid("StatusIcon")
	g_msgid_scoreinfo = get_user_msgid("ScoreInfo")
	g_msgid_scoreattrib = get_user_msgid("ScoreAttrib")
	g_msgid_textmsg = get_user_msgid("TextMsg")
	g_msgid_sendaudio = get_user_msgid("SendAudio")
	g_msgid_teamscore = get_user_msgid("TeamScore")
	g_msgid_money = get_user_msgid("Money")

	register_message(g_msgid_hudtextargs, "message_hudtext_args")
	register_message(g_msgid_statusicon, "message_status_icon")
	register_message(g_msgid_scoreattrib, "message_score_attrib")
	register_message(g_msgid_textmsg, "message_text_msg")
	register_message(g_msgid_teamscore, "message_team_score")

	register_forward(FM_GetGameDescription, "forward_get_game_description")
	register_forward(FM_CreateNamedEntity, "forward_create_named_entity")
	register_forward(FM_Touch, "forward_touch")
	register_forward(FM_AlertMessage, "forward_alert_message")
#if FLAG_AUTORETURN_DELAY > 0
	register_forward(FM_Think, "forward_think")
#endif

	create_bases()

	register_clcmd("fullupdate", "clcmd_fullupdate")
	server_cmd("amx_restrict on defuser")
	set_cvar_num("sv_restartround", 1)
}

create_bases() {
	new ipsz_info_target = engfunc(EngFunc_AllocString, "info_target")
	new ent_base, ent_flag, ent_spawn, classname[] = "classname", Float:vector[3]
	new spawn_class[TA_SIZE][] = {"info_player_deathmatch", "info_player_start"}
	for (new i; i < TA_SIZE; ++i) {
		ent_base = engfunc(EngFunc_CreateNamedEntity, ipsz_info_target)
		set_pev(ent_base, pev_classname, g_ctfbase)
		engfunc(EngFunc_SetSize, ent_base, g_base_min, g_base_max)
		engfunc(EngFunc_SetModel, ent_base, g_mdl_base[g_flag_team[i]])
		set_pev(ent_base, pev_solid, SOLID_TRIGGER)
		set_pev(ent_base, pev_movetype, MOVETYPE_TOSS)
		set_pev(ent_base, pev_framerate, 1.0)
		g_base_ent[i] = ent_base

		ent_flag = engfunc(EngFunc_CreateNamedEntity, ipsz_info_target)
		set_pev(ent_flag, pev_classname, g_ctfflag)
		engfunc(EngFunc_SetSize, ent_flag, g_flag_min, g_flag_max)
		engfunc(EngFunc_SetModel, ent_flag, g_mdl_flag)
		set_pev(ent_flag, pev_solid, SOLID_TRIGGER)
		set_pev(ent_flag, pev_movetype, MOVETYPE_TOSS)
		set_pev(ent_flag, pev_framerate, 1.0)
		set_pev(ent_flag, pev_skin, g_flag_team[i])
		set_pev(ent_flag, pev_sequence, SEQ_BASE)
		g_flag_ent[i] = ent_flag

		ent_spawn = -1
		while ((ent_spawn = engfunc(EngFunc_FindEntityByString, ent_spawn, classname, spawn_class[i]))) {
			pev(ent_spawn, pev_origin, g_base_origin[i])
			engfunc(EngFunc_SetOrigin, ent_base, g_base_origin[i])
			engfunc(EngFunc_DropToFloor, ent_base)
			pev(ent_base, pev_origin, g_base_origin[i])

			engfunc(EngFunc_SetOrigin, ent_flag, g_base_origin[i])
			engfunc(EngFunc_DropToFloor, ent_flag)
			pev(ent_flag, pev_origin, vector)

			if (!get_distance_f(g_base_origin[i], vector))
				break
		}
	}

	vector[0] = g_base_origin[TA_CT][0] - g_base_origin[TA_T][0]
	vector[1] = g_base_origin[TA_CT][1] - g_base_origin[TA_T][1]
	vector[2] = g_base_origin[TA_CT][2] - g_base_origin[TA_T][2]
	engfunc(EngFunc_VecToAngles, vector, vector)
	g_base_angles[TA_T][1] = vector[1]
	g_base_angles[TA_CT][1] = float((floatround(vector[1]) - 180) % 360)

	for (new i; i < TA_SIZE; ++i) {
		set_pev(g_base_ent[i], pev_angles, g_base_angles[i])
		set_pev(g_flag_ent[i], pev_angles, g_base_angles[i])
	}
}

public event_new_game() {
	g_reset_reward = false
}

public event_new_round() {
	flag_to_base()
	if (g_reset_reward) {
		g_reset_reward = false
		set_msg_block(g_msgid_money, BLOCK_SET)
		for (new i = 1; i < g_maxplayers; ++i) {
			if (is_user_connected(i))
				g_user_money[i] = CS_GET_USER_MONEY_(i)
		}
		set_task(0.1, "task_reset_reward")
	}
}

public task_reset_reward() {
	for (new i = 1; i < g_maxplayers; ++i) {
		if (is_user_connected(i)) {
			CS_SET_USER_MONEY_(i, g_user_money[i])
			message_begin(MSG_ONE, g_msgid_money, _, i)
			write_long(g_user_money[i])
			write_byte(0)
			message_end()
		}
	}
	set_msg_block(g_msgid_money, BLOCK_NOT)
}

public logevent_round_end() {
	flag_to_base()
}

public event_reset_hud(id) {
	for (new i = TA_SIZE - 1; i >= 0; --i) {
		message_begin(MSG_ONE, g_msgid_statusicon, _, id)
		write_byte(ICON_NORMAL)
		write_string(g_icon[i])
		for (new j; j < color_num; ++j)
			write_byte(g_icon_color[g_flag_state[i]][j])
		message_end()
	}
}

public event_death() {
	check_and_drop(read_data(2))
}

public client_disconnect(id) {
	check_and_drop(id)
}

public message_hudtext_args() {
	static arg[24]
	get_msg_arg_string(1, arg, 23)
	if (equal(arg, g_have_bomb))
		return PLUGIN_HANDLED

	return PLUGIN_CONTINUE
}

public message_status_icon() {
	if (get_msg_arg_int(1) == ICON_NONE)
		return PLUGIN_CONTINUE

	static arg[4]
	get_msg_arg_string(2, arg, 3)
	if (equal(arg, g_icon_c4))
		return PLUGIN_HANDLED

	return PLUGIN_CONTINUE
}

public message_score_attrib(msg_id, msg_dest, id) {
	if (get_msg_arg_int(2) & ATTRIB_BOMB || get_owner_ta(id) != -1)
		return PLUGIN_HANDLED
	
	return PLUGIN_CONTINUE
}

public message_text_msg() {
	if (get_msg_argtype(2) != ARG_STRING)
		return PLUGIN_CONTINUE

	static arg[24]
	get_msg_arg_string(2, arg, 23)
	for (new i; i < TA_SIZE; ++i) {
		if (equal(arg, g_win_msg[i])) {
			static players[32]
			new num, team = TEAM_FROM_TA(i) == TEAM_T ? TEAM_CT : TEAM_T
			get_players(players, num, "a")

			for (new j; j < num; ++j) {
				if (get_user_teamid(players[j]) == team)
					return PLUGIN_HANDLED
			}

			break
		}
	}

	return PLUGIN_CONTINUE
}

public message_team_score() {
	static arg[16]
	get_msg_arg_string(1, arg, 15)
	new score = get_msg_arg_int(2)
	for (new i; i < TA_SIZE; ++i) {
		if (equal(arg, g_team_name[i])) {
			g_team_score[i] = score
			if (i == TA_CT) {
				if (!score)
					g_ct_scorediff = 0
				else if (g_ct_scorediff)
					set_msg_arg_int(2, ARG_SHORT, score + g_ct_scorediff)
			}

			break
		}
	}
}

public forward_get_game_description() {
	forward_return(FMV_STRING, g_game_description)
	return FMRES_SUPERCEDE
}

public forward_create_named_entity(int_class) {
	static class[16]
	engfunc(EngFunc_SzFromIndex, int_class, class, 15)
	if (equal(class, g_weapon_c4))
		return FMRES_SUPERCEDE

	return FMRES_IGNORED
}

public forward_touch(ent, id) {
	if (!is_user_alive(id))
		return FMRES_IGNORED

	new team = get_user_team(id)
	new ta_team = TA_FROM_TEAM(team)
	new ta_flag = get_flag_ta(ent)
	new ta_base = get_base_ta(ent)

	if (ta_flag != -1) {
		if (ta_team != ta_flag && (g_flag_state[ta_flag] == state_base || g_flag_state[ta_flag] == state_dropped)) {
			set_flag_state(ta_flag, state_carried, id) // taken (base/ground)
			client_cmd(0, "spk ^"%s^"", g_snd_taken[g_flag_team[ta_team]])
			txt_print(txt_taken, id, team)
			add_user_frags(id, BONUS_TAKEN, team)
			msg_set_user_attrib(id, g_attrib[ta_team])
		}
		else if (ta_team == ta_flag && g_flag_state[ta_flag] == state_dropped) {
			set_flag_state(ta_flag, state_base) // returned
			txt_print(txt_returned, id, team)
			add_user_frags(id, BONUS_RETURNED, team)
		}
	}
	else if (ta_team == ta_base) {
		ta_flag = get_owner_ta(id)
		if (ta_flag != -1 && ta_flag != ta_team) {
			if (g_flag_state[ta_team] == state_base) {
				set_flag_state(ta_flag, state_base) // captured

				#if defined FORCE_TEAM_WIN
					cs_force_team_win(CsTeams:team)
				#endif

				client_cmd(0, "spk ^"%s^"", g_snd_captured[g_flag_team[ta_team]])
				txt_print(txt_captured, id, team)
				add_user_frags(id, BONUS_CAPTURED, team)
				msg_set_user_attrib(id, ATTRIB_NONE)

			}
			else {
				new Float:gametime = get_gametime()
				if (gametime > g_next_hint[ta_team]) {
					g_next_hint[ta_team] = gametime + HINT_DELAY
					client_print(id, print_center, g_hint)
				}
			}
		}
	}

	return FMRES_IGNORED
}

public forward_alert_message(at_type, message[]) {
	if (at_type != _:at_logged)
		return FMRES_IGNORED

	if (contain(message, g_spawned_withbomb) != -1) {
		static players[32], num
		get_players(players, num, "ae", g_team_name[TA_T])
		for (new i; i < num; ++i)
			set_pev(players[i], pev_body, 0)

		return FMRES_SUPERCEDE
	}

	if (equal(message, g_target_saved, sizeof(g_target_saved) - 1)) {
		log_message(g_round_draw, g_team_score[TA_CT] + g_ct_scorediff, g_team_score[TA_T])
		g_ct_scorediff--
		g_reset_reward = true

		message_begin(MSG_ALL, g_msgid_textmsg)
		write_byte(print_center)
		write_string(g_draw_msg)
		message_end()

		message_begin(MSG_BROADCAST, g_msgid_sendaudio)
		write_byte(0)
		write_string(g_draw_audio)
		write_short(100)
		message_end()

		return FMRES_SUPERCEDE
	}

	return FMRES_IGNORED
}

public forward_think(ent) {
	new ta_flag = get_flag_ta(ent)
	if (ta_flag != -1) {
		set_flag_state(ta_flag, state_base) // returned (auto)
		txt_print(txt_returned_auto, 0, TEAM_FROM_TA(ta_flag))
	}
}

public clcmd_fullupdate() {
	return PLUGIN_HANDLED
}

flag_to_base() {
	for (new i; i < TA_SIZE; ++i) {
		if (g_flag_state[i] == state_base)
			continue

		if (g_flag_owner[i])
			msg_set_user_attrib(g_flag_owner[i], ATTRIB_NONE)

		set_flag_state(i, state_base)
	}
}

set_flag_state(flag_ta, flag_state, owner = 0) {
	g_flag_state[flag_ta] = flag_state
	g_flag_owner[flag_ta] = owner
	new ent = g_flag_ent[flag_ta]

	switch (flag_state) {
		case state_base: {
			set_pev(ent, pev_solid, SOLID_TRIGGER)
			set_pev(ent, pev_movetype, MOVETYPE_TOSS)
			set_pev(ent, pev_sequence, SEQ_BASE)
			set_pev(ent, pev_nextthink, 0.0)
			engfunc(EngFunc_SetOrigin, ent, g_base_origin[flag_ta])
			set_pev(ent, pev_angles, g_base_angles[flag_ta])
			engfunc(EngFunc_DropToFloor, ent)
		}
		case state_carried: {
			set_pev(ent, pev_solid, SOLID_NOT)
			set_pev(ent, pev_movetype, MOVETYPE_FOLLOW)
			set_pev(ent, pev_sequence, SEQ_CARRIED)
			set_pev(ent, pev_nextthink, 0.0)
		}
		case state_dropped: {
			set_pev(ent, pev_solid, SOLID_TRIGGER)
			set_pev(ent, pev_movetype, MOVETYPE_TOSS)
			set_pev(ent, pev_sequence, SEQ_DROPPED)
			engfunc(EngFunc_DropToFloor, ent)
			#if FLAG_AUTORETURN_DELAY > 0
				set_pev(ent, pev_nextthink, get_gametime() + FLAG_AUTORETURN_DELAY)
			#else
				set_pev(ent, pev_nextthink, 0.0)
			#endif
		}
	}

	set_pev(ent, pev_aiment, owner)
	set_pev(ent, pev_owner, owner)

	message_begin(MSG_ALL, g_msgid_statusicon)
	write_byte(ICON_NORMAL)
	write_string(g_icon[flag_ta])
	for (new i; i < color_num; ++i)
		write_byte(g_icon_color[flag_state][i])
	message_end()
}

check_and_drop(id) {
	new ta_flag = get_owner_ta(id)
	if (ta_flag != -1) {
		#if !FLAG_AUTORETURN_DELAY
			set_flag_state(ta_flag, state_base) // returned (auto)
			txt_print(txt_returned_auto, 0, TEAM_FROM_TA(ta_flag))
		#else
			set_flag_state(ta_flag, state_dropped) // dropped
			txt_print(txt_dropped, id, TEAM_FROM_TA(ta_flag) == TEAM_T ? TEAM_CT : TEAM_T)
		#endif
	}
}

txt_print(txt_id, player, player_team) {
	static players[32]
	new num, id, team
	get_players(players, num)

	for (new i; i < num; ++i) {
		id = players[i]
		team = get_user_teamid(id)
		if (!team)
			continue

		if (id == player)
			client_print(id, print_center, g_txt[txt_id][dest_player])
		else if (team == player_team)
			client_print(id, print_center, g_txt[txt_id][dest_teammates])
		else
			client_print(id, print_center, g_txt[txt_id][dest_enemies])
	}
}

get_user_teamid(id) {
	static tchar[2]
	get_user_team(id, tchar, 1)
	switch (tchar[0]) {
		case 'T': return TEAM_T
		case 'C': return TEAM_CT
	}

	return 0
}

add_user_frags(id, value, team) {
	new Float:frags
	pev(id, pev_frags, frags)
	frags += value
	set_pev(id, pev_frags, frags)

	message_begin(MSG_ALL, g_msgid_scoreinfo)
	write_byte(id)
	write_short(floatround(frags))
	write_short(CS_GET_USER_DEATHS_(id))
	write_short(0)
	write_short(team)
	message_end()
}

msg_set_user_attrib(id, attrib) {
	message_begin(MSG_ALL, g_msgid_scoreattrib)
	write_byte(id)
	write_byte(attrib)
	message_end()
}

get_flag_ta(ent) {
	for (new i; i < TA_SIZE; ++i) {
		if (ent == g_flag_ent[i])
			return i
	}

	return -1
}

get_base_ta(ent) {
	for (new i; i < TA_SIZE; ++i) {
		if (ent == g_base_ent[i])
			return i
	}

	return -1
}

get_owner_ta(id) {
	for (new i; i < TA_SIZE; ++i) {
		if (id == g_flag_owner[i])
			return i
	}

	return -1
}
